using System;
using System.Net;
using System.Reflection;
using System.Text.RegularExpressions;
using System.Windows.Forms;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;
using BrRules = gov.va.med.vbecs.Common.VbecsBrokenRules.VistaLinkConfig;
using gov.va.med.vbecs.GUI;
using gov.va.med.vbecs.GUI.controls;

namespace gov.va.med.vbecs.admin.GUI
{
	/// <summary>
	/// This control encapsulates UI for defining/updating VistALink configuration parameters.
	/// </summary>
	public class EditVistALinkConnectionParameters : BaseAdminFormControl
	{
		private class VistALinkTestResult
		{
			public readonly string Message;
			public readonly int ImageIndex;
			
			private string _errorDetails;
			private bool _isErrorDetailsUpdateable;

			private VistALinkTestResult( StrRes.StringResourceItem message, int imageIndex, bool isErrorDetailsUpdateable, string defaultErrorDetails )
			{
				Message = message.ResString;
				ImageIndex = imageIndex;

				_isErrorDetailsUpdateable = isErrorDetailsUpdateable;
				_errorDetails = defaultErrorDetails;
			}

			public string ErrorDetails
			{
				get
				{
					return _errorDetails;
				}
				set
				{
					if( !_isErrorDetailsUpdateable )
						throw( new InvalidOperationException() );

					_errorDetails = (value != null) ? value : "No error details is available." ;
				}
			}

			static VistALinkTestResult(){} // getting rid of beforefieldinit

			public static readonly VistALinkTestResult 
				NotTested = new VistALinkTestResult( StrRes.OtherMsg.MUC01.TestResultNotTested(), 0, false, StrRes.Tooltips.MUC01.VistALinkConfigWasNotTested().ResString ),
				Testing = new VistALinkTestResult( StrRes.OtherMsg.MUC01.TestResultTesting(), 1, false, StrRes.Tooltips.MUC01.VistALinkConfigIsBeingTested().ResString ),
				Success = new VistALinkTestResult( StrRes.OtherMsg.MUC01.TestResultSuccessful(), 2, false,  StrRes.Tooltips.MUC01.VistALinkConfigTestSucceeded().ResString ),
				Failure = new VistALinkTestResult( StrRes.OtherMsg.MUC01.TestResultFailed(), 3, true, null );
		}

		private const string FORM_NAME = "Configure VistALink";

        readonly ILogger _logger = LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);
		private BOL.VistALinkConfig _config;
		private BusinessObjectGuiValidator _configValidator;

		private VistALinkTestResult _connectionTestResult;
		
		private System.Windows.Forms.GroupBox grpConfigureVistALink;
		private System.Windows.Forms.GroupBox grpMServer;
		private System.Windows.Forms.TextBox txtMServerPort;
		private System.Windows.Forms.TextBox txtMServerIPAddress;
        private System.Windows.Forms.Label lblMServerPort;
        private VbecsButton btnSave;
		private System.Windows.Forms.ImageList imageListVistALinkTestResult;
		private VbecsButton btnClear;
		private System.Windows.Forms.TextBox txtMServerDomain;
		private System.Windows.Forms.RadioButton rdoIpAddress;
		private System.Windows.Forms.RadioButton rdoDomain;
		private System.Windows.Forms.GroupBox grpConnect;
        private GroupBox grpVbecsService;
        private Label lblVbecsServiceIP;
        private Label lblVbecsServicePort;
        private TextBox txtVbecsServicePort;
        private TextBox txtVbecsServiceIpAddress;
        private VbecsButton btnTestConnection;
        private Label lblSuccessIndicator;
        private TextBox txtConnectionTestResults;
		private System.ComponentModel.IContainer components;

		/// <summary>
		/// Constructor
		/// </summary>
		public EditVistALinkConnectionParameters() : base()
		{
			// This call is required by the Windows.Forms Form Designer.
			InitializeComponent();			

			btnSave.Text = "&Save";
			btnClear.Text = "&Clear";

			ResetStateVars();
			BuildVistALinkConfigValidator();
		}

		/// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
            this.components = new System.ComponentModel.Container();
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(EditVistALinkConnectionParameters));
            this.grpConfigureVistALink = new System.Windows.Forms.GroupBox();
            this.grpVbecsService = new System.Windows.Forms.GroupBox();
            this.lblVbecsServiceIP = new System.Windows.Forms.Label();
            this.lblVbecsServicePort = new System.Windows.Forms.Label();
            this.txtVbecsServicePort = new System.Windows.Forms.TextBox();
            this.txtVbecsServiceIpAddress = new System.Windows.Forms.TextBox();
            this.btnClear = new gov.va.med.vbecs.GUI.controls.VbecsButton();
            this.btnSave = new gov.va.med.vbecs.GUI.controls.VbecsButton();
            this.grpMServer = new System.Windows.Forms.GroupBox();
            this.btnTestConnection = new gov.va.med.vbecs.GUI.controls.VbecsButton();
            this.lblSuccessIndicator = new System.Windows.Forms.Label();
            this.grpConnect = new System.Windows.Forms.GroupBox();
            this.txtMServerDomain = new System.Windows.Forms.TextBox();
            this.txtMServerIPAddress = new System.Windows.Forms.TextBox();
            this.rdoIpAddress = new System.Windows.Forms.RadioButton();
            this.rdoDomain = new System.Windows.Forms.RadioButton();
            this.txtMServerPort = new System.Windows.Forms.TextBox();
            this.txtConnectionTestResults = new System.Windows.Forms.TextBox();
            this.lblMServerPort = new System.Windows.Forms.Label();
            this.imageListVistALinkTestResult = new System.Windows.Forms.ImageList(this.components);
            ((System.ComponentModel.ISupportInitialize)(this.erpMainProvider)).BeginInit();
            this.grpConfigureVistALink.SuspendLayout();
            this.grpVbecsService.SuspendLayout();
            this.grpMServer.SuspendLayout();
            this.grpConnect.SuspendLayout();
            this.SuspendLayout();
            // 
            // grpConfigureVistALink
            // 
            this.grpConfigureVistALink.Controls.Add(this.grpVbecsService);
            this.grpConfigureVistALink.Controls.Add(this.btnClear);
            this.grpConfigureVistALink.Controls.Add(this.btnSave);
            this.grpConfigureVistALink.Controls.Add(this.grpMServer);
            this.grpConfigureVistALink.Dock = System.Windows.Forms.DockStyle.Fill;
            this.grpConfigureVistALink.Location = new System.Drawing.Point(0, 0);
            this.grpConfigureVistALink.Name = "grpConfigureVistALink";
            this.grpConfigureVistALink.Size = new System.Drawing.Size(456, 364);
            this.grpConfigureVistALink.TabIndex = 0;
            this.grpConfigureVistALink.TabStop = false;
            this.grpConfigureVistALink.Text = "Configure VistALink";
            // 
            // grpVbecsService
            // 
            this.grpVbecsService.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpVbecsService.Controls.Add(this.lblVbecsServiceIP);
            this.grpVbecsService.Controls.Add(this.lblVbecsServicePort);
            this.grpVbecsService.Controls.Add(this.txtVbecsServicePort);
            this.grpVbecsService.Controls.Add(this.txtVbecsServiceIpAddress);
            this.grpVbecsService.Location = new System.Drawing.Point(8, 207);
            this.grpVbecsService.Name = "grpVbecsService";
            this.grpVbecsService.Size = new System.Drawing.Size(439, 50);
            this.grpVbecsService.TabIndex = 20;
            this.grpVbecsService.TabStop = false;
            this.grpVbecsService.Text = "VBECS VistALink Service";
            // 
            // lblVbecsServiceIP
            // 
            this.lblVbecsServiceIP.Location = new System.Drawing.Point(16, 20);
            this.lblVbecsServiceIP.Name = "lblVbecsServiceIP";
            this.lblVbecsServiceIP.Size = new System.Drawing.Size(69, 20);
            this.lblVbecsServiceIP.TabIndex = 14;
            this.lblVbecsServiceIP.Text = "IP Address*";
            this.lblVbecsServiceIP.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // lblVbecsServicePort
            // 
            this.lblVbecsServicePort.Location = new System.Drawing.Point(277, 20);
            this.lblVbecsServicePort.Name = "lblVbecsServicePort";
            this.lblVbecsServicePort.Size = new System.Drawing.Size(76, 20);
            this.lblVbecsServicePort.TabIndex = 15;
            this.lblVbecsServicePort.Text = "Port Number*";
            this.lblVbecsServicePort.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // txtVbecsServicePort
            // 
            this.txtVbecsServicePort.Location = new System.Drawing.Point(359, 19);
            this.txtVbecsServicePort.MaxLength = 5;
            this.txtVbecsServicePort.Name = "txtVbecsServicePort";
            this.txtVbecsServicePort.Size = new System.Drawing.Size(52, 20);
            this.txtVbecsServicePort.TabIndex = 8;
            this.txtVbecsServicePort.TextChanged += new System.EventHandler(this.txtVbecsServicePort_TextChanged);
            // 
            // txtVbecsServiceIpAddress
            // 
            this.txtVbecsServiceIpAddress.Location = new System.Drawing.Point(90, 19);
            this.txtVbecsServiceIpAddress.MaxLength = 15;
            this.txtVbecsServiceIpAddress.Name = "txtVbecsServiceIpAddress";
            this.txtVbecsServiceIpAddress.Size = new System.Drawing.Size(148, 20);
            this.txtVbecsServiceIpAddress.TabIndex = 7;
            this.txtVbecsServiceIpAddress.TextChanged += new System.EventHandler(this.txtVbecsServiceIpAddress_TextChanged);
            // 
            // btnClear
            // 
            this.btnClear.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
            this.btnClear.EnabledButton = false;
            this.btnClear.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.btnClear.Image = null;
            this.btnClear.Location = new System.Drawing.Point(288, 336);
            this.btnClear.Name = "btnClear";
            this.btnClear.Size = new System.Drawing.Size(75, 23);
            this.btnClear.TabIndex = 1;
            this.btnClear.Click += new System.EventHandler(this.btnClear_Click);
            // 
            // btnSave
            // 
            this.btnSave.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
            this.btnSave.EnabledButton = false;
            this.btnSave.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.btnSave.Image = null;
            this.btnSave.Location = new System.Drawing.Point(372, 336);
            this.btnSave.Name = "btnSave";
            this.btnSave.Size = new System.Drawing.Size(75, 23);
            this.btnSave.TabIndex = 2;
            this.btnSave.Click += new System.EventHandler(this.btnSave_Click);
            // 
            // grpMServer
            // 
            this.grpMServer.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpMServer.Controls.Add(this.btnTestConnection);
            this.grpMServer.Controls.Add(this.lblSuccessIndicator);
            this.grpMServer.Controls.Add(this.grpConnect);
            this.grpMServer.Controls.Add(this.txtMServerPort);
            this.grpMServer.Controls.Add(this.txtConnectionTestResults);
            this.grpMServer.Controls.Add(this.lblMServerPort);
            this.grpMServer.Location = new System.Drawing.Point(8, 16);
            this.grpMServer.Name = "grpMServer";
            this.grpMServer.Size = new System.Drawing.Size(440, 180);
            this.grpMServer.TabIndex = 0;
            this.grpMServer.TabStop = false;
            this.grpMServer.Text = "M Server";
            // 
            // btnTestConnection
            // 
            this.btnTestConnection.EnabledButton = false;
            this.btnTestConnection.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.btnTestConnection.Image = ((System.Drawing.Image)(resources.GetObject("btnTestConnection.Image")));
            this.btnTestConnection.Location = new System.Drawing.Point(11, 138);
            this.btnTestConnection.Name = "btnTestConnection";
            this.btnTestConnection.Size = new System.Drawing.Size(116, 23);
            this.btnTestConnection.TabIndex = 7;
            this.btnTestConnection.Click += new System.EventHandler(this.btnTestConnection_Click);
            // 
            // lblSuccessIndicator
            // 
            this.lblSuccessIndicator.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.lblSuccessIndicator.Location = new System.Drawing.Point(141, 140);
            this.lblSuccessIndicator.Name = "lblSuccessIndicator";
            this.lblSuccessIndicator.Size = new System.Drawing.Size(20, 20);
            this.lblSuccessIndicator.TabIndex = 5;
            this.lblSuccessIndicator.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // grpConnect
            // 
            this.grpConnect.Controls.Add(this.txtMServerDomain);
            this.grpConnect.Controls.Add(this.txtMServerIPAddress);
            this.grpConnect.Controls.Add(this.rdoIpAddress);
            this.grpConnect.Controls.Add(this.rdoDomain);
            this.grpConnect.Location = new System.Drawing.Point(8, 20);
            this.grpConnect.Name = "grpConnect";
            this.grpConnect.Size = new System.Drawing.Size(272, 68);
            this.grpConnect.TabIndex = 5;
            this.grpConnect.TabStop = false;
            this.grpConnect.Text = "Connection Method*";
            // 
            // txtMServerDomain
            // 
            this.txtMServerDomain.Location = new System.Drawing.Point(96, 40);
            this.txtMServerDomain.MaxLength = 100;
            this.txtMServerDomain.Name = "txtMServerDomain";
            this.txtMServerDomain.Size = new System.Drawing.Size(148, 20);
            this.txtMServerDomain.TabIndex = 3;
            this.txtMServerDomain.TextChanged += new System.EventHandler(this.txtMServerDomain_TextChanged);
            // 
            // txtMServerIPAddress
            // 
            this.txtMServerIPAddress.Enabled = false;
            this.txtMServerIPAddress.Location = new System.Drawing.Point(96, 16);
            this.txtMServerIPAddress.MaxLength = 15;
            this.txtMServerIPAddress.Name = "txtMServerIPAddress";
            this.txtMServerIPAddress.Size = new System.Drawing.Size(148, 20);
            this.txtMServerIPAddress.TabIndex = 2;
            this.txtMServerIPAddress.TextChanged += new System.EventHandler(this.txtMServerIPAddress_TextChanged);
            // 
            // rdoIpAddress
            // 
            this.rdoIpAddress.Location = new System.Drawing.Point(16, 16);
            this.rdoIpAddress.Name = "rdoIpAddress";
            this.rdoIpAddress.Size = new System.Drawing.Size(84, 24);
            this.rdoIpAddress.TabIndex = 0;
            this.rdoIpAddress.Text = "IP Address";
            this.rdoIpAddress.CheckedChanged += new System.EventHandler(this.rdo_CheckedChanged);
            // 
            // rdoDomain
            // 
            this.rdoDomain.Location = new System.Drawing.Point(16, 40);
            this.rdoDomain.Name = "rdoDomain";
            this.rdoDomain.Size = new System.Drawing.Size(68, 24);
            this.rdoDomain.TabIndex = 1;
            this.rdoDomain.Text = "Domain";
            this.rdoDomain.CheckedChanged += new System.EventHandler(this.rdo_CheckedChanged);
            // 
            // txtMServerPort
            // 
            this.txtMServerPort.Enabled = false;
            this.txtMServerPort.Location = new System.Drawing.Point(92, 96);
            this.txtMServerPort.MaxLength = 5;
            this.txtMServerPort.Name = "txtMServerPort";
            this.txtMServerPort.Size = new System.Drawing.Size(52, 20);
            this.txtMServerPort.TabIndex = 5;
            this.txtMServerPort.TextChanged += new System.EventHandler(this.txtMServerPort_TextChanged);
            // 
            // txtConnectionTestResults
            // 
            this.txtConnectionTestResults.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.txtConnectionTestResults.Location = new System.Drawing.Point(168, 141);
            this.txtConnectionTestResults.Name = "txtConnectionTestResults";
            this.txtConnectionTestResults.ReadOnly = true;
            this.txtConnectionTestResults.Size = new System.Drawing.Size(251, 20);
            this.txtConnectionTestResults.TabIndex = 5;
            this.txtConnectionTestResults.TabStop = false;
            this.txtConnectionTestResults.Text = "Not tested";
            // 
            // lblMServerPort
            // 
            this.lblMServerPort.Location = new System.Drawing.Point(8, 96);
            this.lblMServerPort.Name = "lblMServerPort";
            this.lblMServerPort.Size = new System.Drawing.Size(76, 20);
            this.lblMServerPort.TabIndex = 1;
            this.lblMServerPort.Text = "Port Number*";
            this.lblMServerPort.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // imageListVistALinkTestResult
            // 
            this.imageListVistALinkTestResult.ImageStream = ((System.Windows.Forms.ImageListStreamer)(resources.GetObject("imageListVistALinkTestResult.ImageStream")));
            this.imageListVistALinkTestResult.TransparentColor = System.Drawing.Color.Transparent;
            this.imageListVistALinkTestResult.Images.SetKeyName(0, "");
            this.imageListVistALinkTestResult.Images.SetKeyName(1, "");
            this.imageListVistALinkTestResult.Images.SetKeyName(2, "");
            this.imageListVistALinkTestResult.Images.SetKeyName(3, "");
            // 
            // EditVistALinkConnectionParameters
            // 
            this.Controls.Add(this.grpConfigureVistALink);
            this.Name = "EditVistALinkConnectionParameters";
            this.Size = new System.Drawing.Size(456, 364);
            this.Load += new System.EventHandler(this.EditVistALinkConnectionParameters_Load);
            ((System.ComponentModel.ISupportInitialize)(this.erpMainProvider)).EndInit();
            this.grpConfigureVistALink.ResumeLayout(false);
            this.grpVbecsService.ResumeLayout(false);
            this.grpVbecsService.PerformLayout();
            this.grpMServer.ResumeLayout(false);
            this.grpMServer.PerformLayout();
            this.grpConnect.ResumeLayout(false);
            this.grpConnect.PerformLayout();
            this.ResumeLayout(false);

		}
		#endregion

		private void ResetStateVars()
		{
			DataLoadMode.Release();
			_connectionTestResult = VistALinkTestResult.NotTested;
		}

		private void EditVistALinkConnectionParameters_Load(object sender, System.EventArgs e)
		{
			btnSave.Text = "&Save";
			btnTestConnection.Text = " &Test Connection";

			this.Parent.Refresh();

#if !DESIGN	
			using( new WaitCursor() )
			{
				UpdateDataEntryGroupState( true );

				DisplayCurrentConfigFromDb();
			}
#endif
		}

		private void BuildVistALinkConfigValidator()
		{
			_configValidator = new BusinessObjectGuiValidator( erpMainProvider, BrRules.FormClassName );
			_configValidator.SetControlBindings( txtMServerPort, BrRules.VistaServerPortNumberNotSet, BrRules.VistaServerPortNumberOutOfRange );
            //CR 3419
            _configValidator.SetControlBindings(txtVbecsServiceIpAddress, BrRules.VbecsServiceIpAddressNotSet);
            _configValidator.SetControlBindings(txtVbecsServicePort, BrRules.VbecsServicePortNumberNotSet, BrRules.VbecsServicePortNumberOutOfRange);
		
        }

		private void DisplayCurrentConfigFromDb()
		{
			_config = BOL.VistALinkConfig.GetCurrent();			

			DataLoadMode.Lock();

			txtMServerPort.Text = StNullConvert.IsNull( _config.VistaPortNumber ) ? String.Empty : _config.VistaPortNumber.ToString();

            // CR 3419
            txtVbecsServiceIpAddress.Text = StNullConvert.IsNull(_config.VbecsServiceIpAddress) ? String.Empty : _config.VbecsServiceIpAddress.ToString();
            txtVbecsServicePort.Text = StNullConvert.IsNull(_config.VbecsServicePortNumber) ? String.Empty : _config.VbecsServicePortNumber.ToString();

			txtMServerIPAddress.Text = StNullConvert.IsNull( _config.VistaIpAddress ) ? String.Empty : _config.VistaIpAddress.ToString();
			if (txtMServerIPAddress.Text.Trim().Length > 0) this.rdoIpAddress.Checked = true;
			txtMServerDomain.Text = StNullConvert.IsNull( _config.VistaDomainName ) ? String.Empty : _config.VistaDomainName.ToString();
			if (txtMServerDomain.Text.Trim().Length > 0) this.rdoDomain.Checked = true;
			

			// if neither checked (initial state), check IP address by default
			if (!rdoIpAddress.Checked && !rdoDomain.Checked)
			this.rdoIpAddress.Checked = true;

			ResetStateVars();
			OnDataChanged(false);
		}

        private void OnDataChanged(bool vistALinkChanged)
		{
            //CR 2936 - removing spaces from IP Address
            if (this.rdoIpAddress.Checked)
            {
                _config.VistaIpAddress = _validator.VerifyIPAddress(txtMServerIPAddress, false) ? StUiNullConvert.ToIPAddress(txtMServerIPAddress.Text.Trim()) : null;
            }

            if (this.rdoDomain.Checked)
            {
                _config.VistaDomainName = StUiNullConvert.ToString(txtMServerDomain.Text, true);
            }

            _config.VistaPortNumber = _validator.VerifyInt32(txtMServerPort, false) ? StUiNullConvert.ToInt32(txtMServerPort.Text) : StNullConvert.Int32Null;

            if (vistALinkChanged) _connectionTestResult = VistALinkTestResult.NotTested;

            // CR 3419
            _config.VbecsServiceIpAddress = _validator.VerifyIPAddress(txtVbecsServiceIpAddress, false) ? StUiNullConvert.ToIPAddress(txtVbecsServiceIpAddress.Text.Trim()) : null;

            _config.VbecsServicePortNumber = _validator.VerifyInt32(txtVbecsServicePort, false) ? StUiNullConvert.ToInt32(txtVbecsServicePort.Text) : StNullConvert.Int32Null;

            // CR 3419 
            _configValidator.Validate(_config);

            UpdateFormControls(vistALinkChanged);				
		}

        private void DataEntryFields_DataChanged(bool VistALinkChanged)
        {
            if (DataLoadMode.IsLocked) // To prevent processing of events when data is being transferred to widgets.
                return;

            OnDataChanged(VistALinkChanged);
        }

		private void btnTestConnection_Click(object sender, System.EventArgs e)
		{
			_connectionTestResult = VistALinkTestResult.Testing;			
			UpdateFormControls(true);

			_connectionTestResult = PerformPingGetResult();
            UpdateFormControls(true);
		}

        //CR3414: added additional handling for wrong server connection info
        //CR3413: moved Ping method inside try/catch block
		private VistALinkTestResult PerformPingGetResult()
		{
			string errorMessage = null;
		    ServerConnectionInfo conn = null;
		    VistALinkTestResult result;
		    try
		    {
		        conn = _config.GetServerConnectionInfo();
                if (DAL.VAL.VistALinkPing.Ping(conn, out errorMessage))
                    return VistALinkTestResult.Success;
		    }
		    catch (Exception ex)
		    {
                result = VistALinkTestResult.Failure;
                Regex rgx = new Regex("(.{57}\\s)");    //CR3419
                string wrappedMessage = rgx.Replace(ex.Message, "$1\n");

                result.ErrorDetails = string.Format("Failed to get server connection.\nError: {0}", wrappedMessage);
                _logger.Error(result.ErrorDetails);
		        return result;
		    }

		    result = VistALinkTestResult.Failure;
            result.ErrorDetails = errorMessage;
            return result;
		}

		private void btnSave_Click(object sender, System.EventArgs e)
		{
            // CR 3419
            if (!Common.Utility.IsApplicationServerIP(this.txtVbecsServiceIpAddress.Text))
            {
                // CR 3419
                if (gov.va.med.vbecs.GUI.GuiMessenger.ShowMessageBox(this, Common.StrRes.ConfMsg.Common.Dynamic("The VBECS VistALink Service IP Address (" + this.txtVbecsServiceIpAddress.Text + ") does not match the server's IP address (" + Common.Utility.GetServerIPv4() + "). This is not a normal configuration. Do you still wish to save?")) != DialogResult.Yes)
                    return;
            }

            if (gov.va.med.vbecs.GUI.GuiMessenger.ShowMessageBox(this, Common.StrRes.ConfMsg.Common.Dynamic("OK to save interface changes? If you have updated the service parameters, the service will also be restarted at this time.")) != DialogResult.Yes)
                return;

            _config.Save(UpdateFunction.MUC01_EditVistALinkConnectionParameters);


            // CR 3419 start: if the services configuration changed, we must restart the services
            if (_config.VistaLinkServiceIsDirty)
            {
                bool success = false;
                if (Common.VbecsConfig.Current.EnvironmentType.ToUpper() == "P")
                {
                    success = Common.Utility.RestartService(true, Common.ApplicationNameGlobalConfig.VistaLinkService);
                }
                else
                {
                    success = Common.Utility.RestartService(false, Common.ApplicationNameGlobalConfig.VistaLinkService);
                }
                if (!success) gov.va.med.vbecs.GUI.GuiMessenger.ShowMessageBox(this, Common.StrRes.SysErrMsg.Common.Dynamic("The service failed to restart. Please contact VBECS support to have the service restarted."));
            }

            

            // CR 3419 end


            DisplayCurrentConfigFromDb();

            if (this._config.VistaLinkConnectionIsDirty)
            {
                // the previous step resets 'success' to 'not tested'.
                _connectionTestResult = VistALinkTestResult.Success;
                UpdateTestResults();
            }

            this._config.VistaLinkConnectionIsDirty = false;
            this._config.VistaLinkServiceIsDirty = false;

            DAL.VAL.VistALink.SetPrimaryServer(_config.GetServerConnectionInfo());
		}

        private void UpdateFormControls(bool vistALinkChanged)
		{
			UpdateSaveButton();
			UpdateClearButton();

            if (vistALinkChanged) // CR 3419
            {
                UpdateTestResults();
                UpdateTestConnectionButton();
            }

			Refresh();
		}

		private void UpdateClearButton()
		{
			UpdateOnFormClearButton( btnClear, _config.IsDirty || _config.IsNew );
		}

		private void UpdateDataEntryGroupState( bool enabled )
		{
			this.grpMServer.Enabled = enabled;
			this.txtMServerPort.Enabled = enabled;
			this.txtMServerIPAddress.Enabled = enabled;
			this.txtMServerDomain.Enabled = enabled;
		}

		private void UpdateTestResults()
		{
			this.txtConnectionTestResults.Text = _connectionTestResult.Message;
			this.lblSuccessIndicator.Image = imageListVistALinkTestResult.Images[ _connectionTestResult.ImageIndex ];			

			SetToolTip( txtConnectionTestResults, _connectionTestResult.ErrorDetails );
			SetToolTip( lblSuccessIndicator, _connectionTestResult.ErrorDetails );
		}

		private void UpdateTestConnectionButton()
		{
            if (_connectionTestResult == VistALinkTestResult.NotTested)
            {
                btnTestConnection.EnabledButton = _config.IsVistALinkConnectionConfigValid; // CR 3419
                btnTestConnection.SetToolTip((_config.IsVistALinkConnectionConfigValid ? StrRes.Tooltips.MUC01.ClickToTestConfig() : StrRes.Tooltips.MUC01.UnableToTestSinceConfigIsInvalid()).ResString);
            }
            else if (_connectionTestResult == VistALinkTestResult.Testing)
            {
                btnTestConnection.EnabledButton = false;
                btnTestConnection.SetToolTip(StrRes.Tooltips.MUC01.VistALinkConfigIsBeingTested().ResString);
            }
            else if (_connectionTestResult == VistALinkTestResult.Success || _connectionTestResult == VistALinkTestResult.Failure)
            {
                btnTestConnection.EnabledButton = true;
                btnTestConnection.SetToolTip(StrRes.Tooltips.MUC01.ClickToTestConfig().ResString);
            }
		}

        // CR 3419: Rewrote this to account for the new fields to support VistALink service.
		private void UpdateSaveButton()
		{
            btnSave.EnabledButton = false;

            if (_config.IsDirty)  // VBECS Services changed
            {
                if (_config.VistaLinkConnectionIsDirty && !_config.VistaLinkTestedSuccessfully) // CR 3419: No need to test again if alread tested successfull and VistALink hasn't changed
                {
                    if (_connectionTestResult == VistALinkTestResult.NotTested)
                    {
                        btnSave.SetToolTip(StrRes.Tooltips.MUC01.ConfigWasNotTestedAfterChange().ResString + "\n" + _config.GetBrokenRulesMessage(BrRules.FormClassName));
                        return;
                    }
                    else if (_connectionTestResult == VistALinkTestResult.Testing)
                    {
                        btnSave.SetToolTip(StrRes.Tooltips.MUC01.VistALinkConfigIsBeingTested().ResString);
                        return;
                    }
                    else if (_connectionTestResult == VistALinkTestResult.Success)
                    {
                        //StrRes.Tooltips.MUC04.NoChangesInConfig()
                        btnSave.SetToolTip((_config.IsDirty ? StrRes.Tooltips.MUC01.ClickToSaveConfig() : StrRes.Tooltips.MUC01.ConfigWasNotChanged()).ResString);
                        _config.VistaLinkTestedSuccessfully = true;
                    }
                    else if (_connectionTestResult == VistALinkTestResult.Failure)
                    {
                        btnSave.SetToolTip(StrRes.Tooltips.MUC01.ConnectionTestFailed().ResString + "\n" + _config.GetBrokenRulesMessage(BrRules.FormClassName));
                        return;
                    }
                }
                //  If you've made it to here, VistAlink is all good. Now, see if there are any issues with the VistALink service.
                if (_config.BrokenRulesCount > 0)
                {
                    btnSave.SetToolTip(_config.GetBrokenRulesMessage(BrRules.FormClassName));
                    return;
                }


                btnSave.EnabledButton = true;
                btnSave.SetToolTip(StrRes.Tooltips.MUC01.ClickToSaveConfig().ResString);
            }
            else // no changes
            {
                btnSave.SetToolTip(StrRes.Tooltips.MUC01.ConfigWasNotChanged().ResString);
            }
		}

		/// <summary>
		/// Get IsDataUnsaved
		/// </summary>
		public bool IsDataUnsaved
		{
			get
			{
				return _config.IsDirty;
			}
		}

		private void btnClear_Click(object sender, EventArgs e)
		{
			DisplayCurrentConfigFromDb();
			//CR 2303
			this.btnClear.EnabledButton = false;

            //CR3419
            UpdateTestResults();
            UpdateTestConnectionButton();
		}

		private void rdo_CheckedChanged(object sender, System.EventArgs e)
		{
			if (this.rdoIpAddress.Checked)
			{
				_configValidator.SetControlBindings( txtMServerIPAddress, BrRules.VistaServerIpAddressAndDomainNameNotSpecified);
				_configValidator.SetControlBindings( txtMServerDomain, string.Empty);
				this.txtMServerIPAddress.Enabled = true;
				this.txtMServerDomain.Enabled = false;
				this._config.VistaDomainName = null; // goes back to dirty here
				this.txtMServerDomain.Text = string.Empty;
			}
			else
			{
				_configValidator.SetControlBindings( txtMServerDomain, BrRules.VistaServerIpAddressAndDomainNameNotSpecified);
				_configValidator.SetControlBindings( txtMServerIPAddress, string.Empty);
				this.txtMServerDomain.Enabled = true;
				this.txtMServerIPAddress.Enabled = false;
				this._config.VistaIpAddress = null;
				this.txtMServerIPAddress.Text = string.Empty;
			}
			OnDataChanged(true);
		}

    

        private void txtMServerIPAddress_TextChanged(object sender, EventArgs e)
        {
            this.DataEntryFields_DataChanged(true);
        }

        private void txtMServerDomain_TextChanged(object sender, EventArgs e)
        {
            this.DataEntryFields_DataChanged(true);
        }

        private void txtMServerPort_TextChanged(object sender, EventArgs e)
        {
            this.DataEntryFields_DataChanged(true);
        }

        private void txtVbecsServiceIpAddress_TextChanged(object sender, EventArgs e)
        {
            this.DataEntryFields_DataChanged(false);
        }

        private void txtVbecsServicePort_TextChanged(object sender, EventArgs e)
        {
            this.DataEntryFields_DataChanged(false);
        }
	}
}

